package com.vaadin.copilot.ai;

import java.util.List;
import java.util.Map;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.copilot.UIServiceCreator;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.flow.internal.JacksonUtils;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

public class AICreateFieldsHandler extends CopilotCommand {

    private final CopilotServerClient copilotServerClient = new CopilotServerClient();

    public record CreateBeanFieldsData(String name, String description) {

    }

    public record CreateBeanFieldsRequest(CreateBeanFieldsData data,
            CopilotServerClient.AccessControlData accessControlData) {
    }

    public record CreateBeanFieldsResponse(List<UIServiceCreator.FieldInfo> fieldInfos) {

    }

    public record CreateExampleDataData(String entityName, List<UIServiceCreator.FieldInfo> fieldInfos, int count) {
    }

    public record CreateExampleDataRequest(CreateExampleDataData data,
            CopilotServerClient.AccessControlData accessControlData) {
    }

    public record CreateExampleDataResponse(List<Map<String, String>> exampleData) {
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("ai-create-bean-fields")) {
            UserInfoServerClient.throwIfAIUsageDisabled();
            ObjectNode respData = JacksonUtils.createObjectNode();
            respData.put(CopilotCommand.KEY_REQ_ID, data.get(CopilotCommand.KEY_REQ_ID).asText());

            CreateBeanFieldsRequest request = new CreateBeanFieldsRequest(
                    new CreateBeanFieldsData(data.get("name").asText(), data.get("description").asText()),
                    CopilotServerClient.AccessControlData.create());

            copilotServerClient.sendCopilotRequest("create-bean-fields", request, CreateBeanFieldsResponse.class,
                    (response) -> {
                        respData.set("fieldInfos", JacksonUtils.listToJson(response.fieldInfos()));

                        devToolsInterface.send(command + "-resp", respData);
                    }, devToolsInterface, command, respData);

            return true;
        } else if (command.equals("ai-generate-example-data")) {
            ObjectNode respData = JacksonUtils.createObjectNode();
            respData.put(CopilotCommand.KEY_REQ_ID, data.get(CopilotCommand.KEY_REQ_ID).asText());
            List<UIServiceCreator.FieldInfo> fieldInfos = JacksonUtils.stream(data.withArray("fieldInfos"))
                    .map(UIServiceCreator.FieldInfo::fromJson).toList();
            CreateExampleDataData exampleDataData = new CreateExampleDataData(data.get("entityName").asText(),
                    fieldInfos, 10);
            copilotServerClient.sendCopilotRequest("create-example-data",
                    new CreateExampleDataRequest(exampleDataData, CopilotServerClient.AccessControlData.create()),
                    CreateExampleDataResponse.class, exampleDataResponse -> {
                        List<Map<String, String>> exampleData = exampleDataResponse.exampleData();
                        respData.set("exampleData", JacksonUtils.listToJson(exampleData));
                        devToolsInterface.send(command + "-resp", respData);
                    }, devToolsInterface, command, respData);
            return true;
        }
        return false;

    }

}
