package com.vaadin.copilot;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.customcomponent.CustomComponentAddMethodInfo;
import com.vaadin.copilot.customcomponent.CustomComponentDetector;
import com.vaadin.copilot.customcomponent.CustomComponentHelper;
import com.vaadin.copilot.customcomponent.CustomComponentResponseData;
import com.vaadin.copilot.exception.ComponentCreatedInLoopException;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.SourceSyncChecker;
import com.vaadin.copilot.plugins.propertypanel.ComponentProperty;
import com.vaadin.copilot.plugins.propertypanel.ComponentPropertyProvider;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasComponents;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.JacksonUtils;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles component specific commands
 */
public class ComponentInfoHandler extends CopilotCommand {
    private static final String ERROR_JSON_KEY = "error";
    private static final String ERROR_MSG_JSON_KEY = "errorMessage";

    private final ComponentSourceFinder componentSourceFinder;
    private final SourceSyncChecker sourceSyncChecker;

    public ComponentInfoHandler(SourceSyncChecker sourceSyncChecker) {
        componentSourceFinder = new ComponentSourceFinder(getVaadinSession());
        this.sourceSyncChecker = sourceSyncChecker;
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-component-source-info")) {
            // command that is called to get information about components after page is
            // loaded.
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());

            int uiId = data.get("uiId").asInt();

            Map<Component, ComponentTypeAndSourceLocation> allComponents = FlowUtil
                    .findAllComponents(getVaadinSession(), uiId);
            for (ComponentTypeAndSourceLocation componentTypeAndSourceLocation : allComponents.values()) {
                sourceSyncChecker.throwIfViewDeleted(componentTypeAndSourceLocation);
            }

            List<Component> componentsInProject = allComponents.entrySet().stream().filter(entry -> {
                Optional<ComponentTracker.Location> create = entry.getValue().createLocationInProject();
                return create.filter(location -> getProjectFileManager().getSourceFile(location).exists()).isPresent();
            }).map(Map.Entry::getKey).toList();

            componentsInProject.stream().map(Component::getClass).collect(Collectors.toSet())
                    .forEach(clazz -> CustomComponentDetector.detectAndPutIfPresent(getVaadinSession(), clazz));

            CustomComponentResponseData customComponentResponseData = CustomComponentHelper
                    .generateResponse(getVaadinSession(), allComponents, componentsInProject);

            responseData.set("dragDropApiInfos",
                    JacksonUtils.writeValue(this.getComponentsDragDropApiMap(componentsInProject)));
            responseData.set("customComponentResponse", JacksonUtils.writeValue(customComponentResponseData));
            responseData.set("nodeIdsInProject",
                    JacksonUtils.listToJson(componentsInProject.stream().map(FlowUtil::getNodeId).toList()));
            devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            return true;
        } else if (command.equals("get-properties")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());
            try {
                int uiId = data.get("uiId").asInt();
                int nodeId = data.get("nodeId").asInt();
                ComponentPropertyProvider componentPropertyProvider = new ComponentPropertyProvider(
                        componentSourceFinder);
                Optional<Component> componentOptional = FlowUtil.findComponentByNodeIdAndUiId(getVaadinSession(),
                        nodeId, uiId);
                if (componentOptional.isPresent()) {
                    List<ComponentProperty> properties = componentPropertyProvider
                            .getProperties(componentOptional.get());
                    responseData.set("properties", JacksonUtils.listToJson(properties));
                    responseData.put(ERROR_JSON_KEY, false);
                } else {
                    responseData.put(ERROR_MSG_JSON_KEY, "Unable to find component with id " + nodeId);
                    responseData.put(ERROR_JSON_KEY, true);
                }
            } catch (ComponentCreatedInLoopException e) {
                responseData.put(ERROR_MSG_JSON_KEY, "Component instantiated in a loop");
                responseData.put(ERROR_JSON_KEY, true);
            } catch (Exception e) {
                getLogger().warn("Unable to get properties for {} ", command, e);
                responseData.put(ERROR_JSON_KEY, true);
            }
            devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            return true;
        } else if (command.equals("get-child-accepting-methods")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());
            int uiId = data.get("uiId").asInt();
            int nodeId = data.get("nodeId").asInt();
            // filter by class hierarchy

            ComponentTypeAndSourceLocation typeAndSourceLocation = componentSourceFinder.findTypeAndSourceLocation(uiId,
                    nodeId, false);

            List<String> classHierarchy = CustomComponentHelper
                    .getClassHierarchyOfJavaClassNameOrReactTagFromRequest(data);
            Component component = FlowUtil.findComponentByNodeIdAndUiId(getVaadinSession(), nodeId, uiId)
                    .orElseThrow(() -> new IllegalArgumentException("Unable to find component with id " + nodeId));
            List<JavaReflectionUtil.ComponentAddableMethod> childAddableMethods = JavaReflectionUtil
                    .getChildAddableMethods(component.getClass());

            List<CustomComponentAddMethodInfo> filteredMethods = CustomComponentHelper
                    .filterMethodBasedOnDraggedComponentHierarchy(childAddableMethods, classHierarchy).stream()
                    .map(componentAddableMethod -> new CustomComponentAddMethodInfo(componentAddableMethod.className(),
                            componentAddableMethod.methodName(), componentAddableMethod.paramJavaClassName()))
                    .toList();
            if (filteredMethods.isEmpty()) {
                // All methods are filtered out. There is no need to fetch component information
                // to get replace, add options
                responseData.set("methods", JacksonUtils.listToJson(filteredMethods));
                devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
                return true;
            }

            try {
                CustomComponentHelper.addInfoFromComponentSource(typeAndSourceLocation, filteredMethods);
            } catch (IOException e) {
                responseData.put(ERROR_JSON_KEY, true);
                responseData.put(ERROR_MSG_JSON_KEY, "An error occurred while reading component info");
                devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
                return true;
            }

            responseData.set("methods", JacksonUtils.listToJson(filteredMethods));
            devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);

            return true;
        } else if (command.equals("is-node-source-present")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());
            int uiId = data.get("uiId").asInt();
            int nodeId = data.get("nodeId").asInt();
            // filter by class hierarchy
            try {
                componentSourceFinder.findTypeAndSourceLocation(uiId, nodeId, false);
                responseData.put("presence", true);
            } catch (Exception e) {
                responseData.put("presence", false);
            }

            devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            return true;
        } else if (command.equals("get-wrap-with-component-list")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());
            WrapWithComponentListFinder wrapWithComponentListFinder = new WrapWithComponentListFinder(
                    getVaadinContext(), getVaadinSession());
            List<WrapWithComponentListFinder.ClassNameInfo> classNameInfos = wrapWithComponentListFinder.findAndGet();
            ArrayNode jsonNodes = JacksonUtils.listToJson(classNameInfos);
            responseData.set("components", jsonNodes);
            devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            return true;
        }

        return false;
    }

    /**
     * A record that encapsulates metadata about a component's drag-and-drop API
     * support.
     * <p>
     * This data structure is used to indicate the drag-and-drop capabilities and
     * child component management features of a specific component in a Vaadin
     * application.
     *
     * @param acceptingChildren
     *            whether the component can accept child components.
     * @param hasAddMethod
     *            whether the component provides an <code>add</code> method.
     * @param hasOnlyAddMethod
     *            whether the component has only the <code>add</code> method (i.e.,
     *            lacks alternatives like <code>addComponent</code>,
     *            <code>setContent</code>, etc.).
     * @param methodSelectionDialogAvailable
     *            whether a method selection dialog should be shown for adding
     *            children.
     * @param implementingHasComponents
     *            whether the component implements the {@code HasComponents}
     *            interface.
     */
    public record ComponentDragDropApiInfo(boolean acceptingChildren, boolean hasAddMethod, boolean hasOnlyAddMethod,
            boolean methodSelectionDialogAvailable, boolean implementingHasComponents) {
    }

    private Map<Integer, ComponentDragDropApiInfo> getComponentsDragDropApiMap(List<Component> allComponents) {
        Map<Integer, ComponentDragDropApiInfo> map = new HashMap<>();
        for (Component component : allComponents) {
            Integer nodeId = FlowUtil.getNodeId(component);
            List<JavaReflectionUtil.ComponentAddableMethod> childAddableMethods = JavaReflectionUtil
                    .getChildAddableMethods(component.getClass());
            boolean hasAddMethod = childAddableMethods.stream()
                    .anyMatch(componentAddableMethod -> componentAddableMethod.methodName().equals("add")
                            && componentAddableMethod.className().contains("HasComponents"));
            boolean acceptingChildren = !childAddableMethods.isEmpty();
            boolean hasOnlyAddMethod = hasAddMethod && childAddableMethods.size() == 1;
            boolean hasApiDialog = acceptingChildren && !hasOnlyAddMethod;
            map.put(nodeId, new ComponentDragDropApiInfo(acceptingChildren, hasAddMethod, hasOnlyAddMethod,
                    hasApiDialog, component instanceof HasComponents));
        }
        return map;
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
