/*
 * Copyright 2000-2025 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.vaadin.pro.licensechecker;

/**
 * Thrown to indicate that a required license key is missing. This exception is
 * used to inform that the operation cannot be completed due to the absence of a
 * valid license key.
 */
public class MissingLicenseKeyException extends LicenseException {

    private final String productName;
    private final String productVersion;

    /**
     * Constructs a new MissingLicenseKeyException with a specified detail
     * message and product information. This exception denotes that a license
     * key is required but is missing for the provided {@code product}.
     *
     * @param product
     *            the product requiring a license key, which contains the
     *            product name and version
     * @param message
     *            the detail message detailing the exception context, can be
     *            {@literal null}
     */
    public MissingLicenseKeyException(Product product, String message) {
        super(buildMessage(product, message));
        this.productName = product.getName();
        this.productVersion = product.getVersion();
    }

    /**
     * Constructs a new MissingLicenseKeyException for the specified product.
     * This exception denotes that a license key is required but is missing for
     * the provided {@code product}.
     *
     * @param product
     *            the product requiring a license key, which contains the
     *            product name and version
     */
    public MissingLicenseKeyException(Product product) {
        this(product, null);
    }

    /**
     * Constructs a new MissingLicenseKeyException with the specified detail
     * message. This exception denotes that a license key is required but is
     * missing.
     *
     * @param message
     *            the detail message providing information about the exception,
     *            which can be null
     */
    public MissingLicenseKeyException(String message) {
        super(message);
        this.productName = null;
        this.productVersion = null;
    }

    private static String buildMessage(Product product, String message) {
        String out = "A license is required to use " + product.getName() + " "
                + product.getVersion();
        if (message != null) {
            out += ". " + message;
        }
        return out;
    }

    /**
     * Gets the name of the product associated with this exception.
     *
     * @return the name of the product.
     */
    public String getProductName() {
        return productName;
    }

    /**
     * Gets the version of the product associated with this exception.
     *
     * @return the product version.
     */
    public String getProductVersion() {
        return productVersion;
    }

}
