/*
 * SonarQube Scanner for Gradle
 * Copyright (C) 2015-2025 SonarSource
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonarqube.gradle;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.gradle.api.DefaultTask;
import org.gradle.api.file.FileCollection;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.InputFiles;
import org.gradle.api.tasks.Optional;
import org.gradle.api.tasks.OutputFile;
import org.gradle.api.tasks.TaskAction;

public abstract class SonarResolverTask extends DefaultTask {
  public static final String TASK_NAME = "sonarResolver";
  public static final String TASK_DESCRIPTION = "Resolves and serializes project information and classpath for SonarQube analysis.";
  private static final Logger LOGGER = Logger.getLogger(SonarResolverTask.class.getName());

  private String projectName;
  private boolean isTopLevelProject;
  private FileCollection compileClasspath;
  private FileCollection testCompileClasspath;
  private File outputDirectory;

  @Input
  public String getProjectName() {
    return projectName;
  }

  public void setProjectName(String name) {
    this.projectName = name;
  }

  @Input
  public boolean isTopLevelProject() {
    return isTopLevelProject;
  }

  public void setTopLevelProject(boolean topLevelProject) {
    this.isTopLevelProject = topLevelProject;
  }

  @InputFiles
  @Optional
  FileCollection getCompileClasspath() {
    return this.compileClasspath;
  }

  public void setCompileClasspath(FileCollection compileClasspath) {
    this.compileClasspath = compileClasspath;
  }

  @InputFiles
  @Optional
  FileCollection getTestCompileClasspath() {
    return this.testCompileClasspath;
  }

  public void setTestCompileClasspath(FileCollection testCompileClasspath) {
    this.testCompileClasspath = testCompileClasspath;
  }

  public void setOutputDirectory(File outputDirectory) {
    this.outputDirectory = outputDirectory;
  }

  @OutputFile
  public File getOutputFile() throws IOException {
    String filename = "properties";
    File output = new File(outputDirectory, filename);
    if (output.isFile() && output.exists()) {
      return output;
    }
    if (!output.createNewFile()) {
      throw new IOException("Could not create output file: " + output.getAbsolutePath());
    }
    return output;
  }

  @TaskAction
  void run() throws IOException {
    String displayName = getProjectName();
    if (LOGGER.isLoggable(Level.INFO)) {
      LOGGER.info("Resolving properties for " + displayName + ".");
    }

    // If we failed to initialize class paths at configuration time AND the configuration cache is not active/requested,
    // we attempt to rebuild them using the source sets.
    if (compileClasspath == null && configurationCacheIsDisabled()) {
      compileClasspath = SonarUtils.getMainClassPath(getProject());
    }
    if (testCompileClasspath == null && configurationCacheIsDisabled()) {
      testCompileClasspath = SonarUtils.getTestClassPath(getProject());
    }

    List<String> compileClasspathFilenames = SonarUtils.exists(compileClasspath == null ? Collections.emptyList() : compileClasspath)
      .stream()
      .map(File::getAbsolutePath)
      .collect(Collectors.toList());
    List<String> testCompileClasspathFilenames = SonarUtils.exists(testCompileClasspath == null ? Collections.emptyList() : testCompileClasspath)
      .stream()
      .map(File::getAbsolutePath)
      .collect(Collectors.toList());
    ProjectProperties projectProperties = new ProjectProperties(getProjectName(), isTopLevelProject(), compileClasspathFilenames, testCompileClasspathFilenames);

    ResolutionSerializer.write(
      getOutputFile(),
      projectProperties
    );
    if (LOGGER.isLoggable(Level.INFO)) {
      LOGGER.info("Resolved properties for " + displayName + " and wrote them to " + getOutputFile() + ".");
    }
  }

  public boolean configurationCacheIsDisabled() {
    return true;
  }

}
